#!/usr/local/bin/perl
# 
# $Header: has/install/crsconfig/oraacfs.pm /st_has_12.1.0.1/1 2013/01/05 07:41:15 gsanders Exp $
#
# oraacfs.pm
# 
# Copyright (c) 2008, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      oraacfs.pm - Library module for ACFS root install functions.
#
#    DESCRIPTION
#      oraacfs.pm - Contains initial installation and deinstallation
#                   routines for ACFS.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    gsanders    12/21/12 - Bug 16007983 register with volume disabled
#    gsanders    11/26/12 - upgrade_acfs_registry return if ACFS not supported.
#    madoming    10/31/12 - Function actionACFSDriversResource should return
#                           SUCCESS when ACFS Drivers are not supported.
#    madoming    10/22/12 - Return SUCCESS explicitly in an development 
#                           environment.
#    madoming    09/20/12 - If we can not remove ACFS Drivers, we'll turn off
#                           the ACFS persistent log.
#    madoming    09/18/12 - Add a error message when fail to uninstall acfs.
#    madoming    08/28/12 - Bug 14476864 - Failed to stop current oracle 
#                           clusterware stack during upgrade, add is_dev_env
#                           check.
#    madoming    08/20/12 - Bug 14323082 - During delete resource 
#                           ora.drivers.acfs, add by default -f attribute.
#    ifarfan     08/15/12 - Change '-' for '/' while executing acfsutil on Windows.
#    agraves     05/11/12 - Remove LASTNODE check in deleteACFSRegistry, as the
#                           function upgrade_acfs_registry is only called
#                           during last node processing.
#    anjiwaji    05/08/12 - Rename createACFSDriversResource to
#                           actionACFSDriversResource and add resource
#                           delete action
#    gsanders    03/24/12 - upgrade_acfs_registry retcode on -l V11 failure
#    anjiwaji    03/09/12 - Fix SIHA check
#    gsanders    03/08/12 - upgrade_acfs_registry return code LRG 6804883
#    anjiwaji    02/29/12 - Handle the case for Windows in drivers resource
#                           creation, Bug 13785935
#    anjiwaji    02/15/12 - Add reboot logic to removeACFSRoot
#    gsanders    01/30/12 - remove removeACFSRegistry()
#    gsanders    01/25/12 - Upgrade support 11.x to 12c
#    anjiwaji    12/14/11 - Add function to create/start the ACFS drivers.
#    anjiwaji    01/19/12 - Fix message thrown by installUSMDriver
#    anjiwaji    01/17/12 - Remove debug lines
#    agraves     11/30/11 - Change reboot necessary to be a hard failure and
#                           disable CRS.
#    gsanders    12/14/11 - ACFS registry OCR to CRS upgrade
#    gsanders    10/27/11 - Fix bug 13247676
#    agraves     11/09/11 - Remove leading 0 from 9427 message check.
#    gsanders    11/05/11 - Fix bug 13263435, look for msg ACFS-9427
#    xyuan       08/02/11 - Incorporate changes from agraves_bug-12788726
#    xyuan       07/27/11 - XbranchMerge xyuan_bug-12701521 from
#                           st_has_11.2.0.3.0
#    anjiwaji    06/14/11 - Bug12644077 Update registry ACL attrib if it
#                           already exists.
#    dpham       04/20/11 - Move acfs' related functions from crsdeconfig
#    agraves     04/07/11 - Update the checking of return codes from crsctl, to
#                           use output. crsctl returns 0 usually.
#    agraves     03/07/11 - Initial Creation
# 

=head1 NAME

  oraacfs.pm  Oracle clusterware ACFS component configuration/startup package

=head1 DESCRIPTION

   This package contains functions required for initial configuration
   and startup of the ACFS component of Oracle clusterware

=cut

package oraacfs;
use strict;
use English;
use File::Temp qw/ tempfile /;
use File::Spec::Functions;
use File::Find ();

use crsohasd;
use crsutils;

use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);
@ISA = qw(Exporter);

my @exp_func  = qw(installUSMDriver
                   disableACFSDriver
                   deleteACFSDriver
                   upgrade_acfs_registry
                   deleteACFSRegistry
                   removeACFSRoot 
                   acfsrepl_updateResources
                   acfsrepl_lastNode
                   actionACFSDriversResource);

# This should export ACFS SUPPORTED constants.
#my @exp_const = qw(CSS_EXCL_SUCCESS CSS_EXCL_FAIL
#                   CSS_EXCL_FAIL_CLUSTER_ACTIVE);

push @EXPORT, @exp_func; #, @exp_const;


# OSD API definitions
# TODO: We'll need this for acfs if there are OSD sections.
#use s_oraacfs;



=head2 upgrade_acfs_registry

   Upgrades the ACFS Registry

=head3 Parameters

   None

=head3 Returns

  TRUE  - ACFS Registry successfully upgraded.
  FALSE - ACFS Registry not successfully upgraded.

=head3 Notes


=cut
sub upgrade_acfs_registry
{
    my $cmd;
    my $status;
    my $srvctlBin = catfile ($CFG->ORA_CRS_HOME, "bin", "srvctl");
    my $acfsutil;
    if ($CFG->platform_family eq 'windows') {
        $acfsutil = catfile ($CFG->ORA_CRS_HOME, 'bin', 'acfsutil');
    }
    else {
      $acfsutil = "/sbin/acfsutil";
    }
    my @ocrRegList;

    #
    #  If USM isn't supported, return.
    #

    return SUCCESS if ( ! isACFSSupported() );

    #
    #  Get list of file systems registered in OCR pre 12.X
    #

    $cmd = "$acfsutil registry -l -V 11";
    if ($CFG->platform_family eq 'windows') {
        $cmd =~ s/\ -/\ \//g ;
    }
    ($status, @ocrRegList) = system_cmd_capture($cmd);
    if ($status != 0)
    {
      trace ("$cmd ... failed");
      print_error(422);
      return FAILED;
    }

    #
    #  For each file system registered in the OCR reregister it with
    #  CRS.
    #

    foreach my $ocrReg ( @ocrRegList )
    {
        my $device      = "";
        my $path        = "";
        my $options     = "";
        my $optionsArg  = "";
        my $nodes       = "";
        my $nodeArg     = "";
        my $diskGroup   = "";
        my $volume      = "";
        my @out;

        chomp $ocrReg;

        #
        #  Format hint:
        #       Device : /dev/asm/crsdg1vol5-326 : Mount Point :
        #       /tkfv_mounts/crsdg1vol5 : Options : none : Nodes : all :
        #       Disk Group : CRSDG1 : Volume : CRSDG1VOL5
        #  We could (and are capable of) writing much more robust
        #  regular expressions here but we're not going to.
        #

        $device = $ocrReg;
        $device =~ s/^Device : (.*) : Mount Point : .*/$1/;

        $path = $ocrReg;
        $path =~ s/.* : Mount Point : (.*) : Options : .*/$1/;

        $options = $ocrReg;
        $options =~ s/.* : Options : (.*) : Nodes : .*/$1/;
        $optionsArg = "-o $options" if ( $options ne "none" );

        $nodes = $ocrReg;
        $nodes =~ s/.* : Nodes : (.*) : Disk Group : .*/$1/;
        $nodeArg = "-n $nodes" if ( $nodes ne "all" );

        $diskGroup = $ocrReg;
        $diskGroup =~ s/.* : Disk Group : (.*) : Volume : .*/$1/;

        $volume = $ocrReg;
        $volume =~ s/.* : Volume : (.*)/$1/;

        #  We have all the attrs from this registry necessary to create
        #  the new CRS Resource based registration so create it.
        #
        #  But first a work around. As of 12.1 ASM volume devices, on
        #  which file systems reside, have a corresponding CRS
        #  resource.  Normally this would be created when the volume
        #  was enabled.  But since the volume was enabled in a
        #  pre-12.1 stack the resource didn't get created. Create it
        #  here and now.  Note that if the upgrade is being rerun the
        #  volume might already exist.

        $cmd = "$srvctlBin status volume -d $device";
        ($status, @out) = system_cmd_capture($cmd);
        if ( scalar(grep(/PRCA-1051/, @out) ) )
        {
          # The volume doesn't exist.  Create it.
          $cmd = "$srvctlBin add volume " .
                 "-g $diskGroup " .
                 "-d $device " .
                 "-volume $volume";
          trace("Invoking \"$cmd\"");
          if ( system_cmd ( $cmd ) )
          {
            # Error creating the volume
            print_error(418, $device, $path );
            return FAILED;
          }
        }
        elsif ( $status != 0 )
        {
          # Some error occurred attempting to run srvctl
          print_error(419);
          return FAILED;
        }
        else
        {
          # No errors detected in srvctl output from volume status
          # command.  That must mean the volume already esists for some
          # reason.  Probably the upgrade is being rerun.
          trace ("$device already esists (OK)");
        }

        #  OK. Let's register the file system in CRS if it doesn't exist
        #  already. (It might already exist if the upgrade is being rerun.)

        $cmd = "$srvctlBin status filesystem -d $device";
        ($status, @out) = system_cmd_capture($cmd);
        if ( scalar(grep(/PRCA-1070/, @out)) )
        {
          $cmd = "$acfsutil registry -v -a " .
                 "$nodeArg $optionsArg " .
                 "$path $device";
          if ($CFG->platform_family eq 'windows') {
              $cmd =~ s/\ -/\ \//g ;
          }
          ($status, @out) = system_cmd_capture($cmd);
          if ( $status != 0 )
          {
            if ( scalar(grep(/PRCA-1089/, @out)) )
            {
              # Bug 16007983 acfsutil calls srvctl to register. srvctl
              # couldn't get the volume info.  Maybe volume is disabled.
              # Forge onward.
              print_error(420, $device, $path );
            } 
            else
            {
              # fatal error registering volume with srvctl.
              print_error(420, $device, $path );
              return FAILED;
            }
          }
        }
        elsif ( $status != 0 )
        {
          # Some error occurred attempting to run srvctl
          print_error(421);
          return FAILED;
        }
        else
        {
          # No errors detected in srvctl output from volume status
          # command.  That must mean the volume already esists for some
          # reason.  Probably the upgrade is being rerun.
          trace ("$device already registered (OK)");
        }
    }

    #
    #  Delete the old registry resource, ora.registry.acfs. It is no
    #  longer used to manage the registered file system.
    #

    if ( deleteACFSRegistry() != SUCCESS )
    {
      return FAILED;
    }
    return SUCCESS;
}

=head2 installUSMDriver

   Installs ACFS, ADVM, OKS - kernel and user components via acfsroot.

=head3 Parameters

   None

=head3 Returns

  TRUE  - Configuration successful
  FALSE - Configuration failed

=head3 Notes


=cut
sub installUSMDriver
{
   my $acfsroot;
   my $ret = SUCCESS;
   my $crsctl = crs_exec_path('crsctl');
   my ($has) = @_;

   # if we are running in development mode, then limit support to only when
   # the appropriate env variables are set

   if (is_dev_env())
   {
      my $acfsInstall = uc($ENV{'USM_ENABLE_ACFS_INSTALL'});

      # if this ENV is not set then we give up early
      if ( $acfsInstall ne "TRUE" )
      {
         trace("ADVM/ACFS disabled because of ENV in test mode");
         return $ret;
      }
   }

   if ($CFG->platform_family eq 'windows') {
      $acfsroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'acfsroot.bat');
   }
   else {
      $acfsroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'acfsroot');
   }

   if (-e $acfsroot) {
      my $cmd = "$acfsroot install";
      
      trace("Executing '$cmd'");
      my @output = system_cmd_capture($cmd);
      my $rc = shift @output;

      if ($rc == 0) {
         my $enable = "$crsctl enable $has";
         my @output = system_cmd_capture($enable);
         my $rc = shift @output;
         trace ("$cmd ... success");

         if ($rc != 0) {
           trace("$enable .... unable to enable CRS.");
           print_error(412);
           $ret = FAILED;
         }
         else {
           trace("$enable .... success.");
         }
      }
      elsif ($rc != 2) {
         # 9427 covers unload failures.
         # 9428 covers install and load failures.
         #   (driver currently running, driver in use can't copy file,
         #    driver unable to load because Windows thinks its running
         #    but its not.)
         if ((scalar(grep(/9427/, @output))) || 
             (scalar(grep(/9428/, @output))) > 0)
         {
           #We'll keep going if we need to reboot but we are in a development
           #environment.
           if (is_dev_env())
           {
              trace ("Skipping reboot in development env ... success");
              return SUCCESS;
           }

           $ret = REBOOT;
           my $disable = "$crsctl disable $has";
           trace("$disable ... disabling CRS in preparation for reboot.");
           my @output = system_cmd_capture($disable);
           my $rc = shift @output;
           if ($rc != 0)
           {
             trace("$disable ... unable to disable CRS for reboot.");
             # Don't fail here - the user should correct
             # the disable, by running it by hand, and reboot.
             print_error(413);
           }
           else
           {
             trace("$disable ... CRS disabled, ready for reboot.");
             # At this point, CRS is disabled.  The user will need
             # to reboot, rerun the operation.  This will rely on 
             # OPatch and OUI to print the appropriate error message.
           }
         }
        else
         {
           $ret = FAILED;
           trace ("$cmd ... failed");
         }
      }
   }
   else {
      trace("$acfsroot not found");
      if (isACFSSupported())
      {
        # if acfsroot not found and usm supported, we have a problem
        # some required files are not here.
        trace("ACFS is supported on this platform, but install files are missing.");
        $ret = FAILED;
      }
      else
      {
        # If acfsroot not found and acfs not supported,
        # then assume everything is okay.
        trace("ACFS is not supported on this platform.");
        $ret = SUCCESS;
      }
   }

  trace("USM driver install status is $ret");
  return $ret;
}

sub deleteACFSRegistry
#-------------------------------------------------------------------------------
# Function: Delete ACFS Registry resource.
# Args    : 0
#-------------------------------------------------------------------------------
{
   my $node   = $CFG->HOST;
   my $crsctl = crs_exec_path('crsctl');
   my $res    = 'ora.registry.acfs';
   my @out;
   my $status;
   my @cmd;

   # check if acfs registry status
   @cmd = ($crsctl, 'stat', 'res', $res );
   @out      = system_cmd_capture(@cmd);
   if ( "@out" =~ "CRS-2613" ) { # Could not find resource 'ora.registry.acfs'
      trace ("ACFS registry resource does not exist. (OK)");
      return SUCCESS;
   }

   # delete acfs registry only called if lastnode. 
   # Note we don't stop it.  Stopping
   # it would unmount the file systems and in a upgrade to 12.1 we don't
   # want to unmount them.  The -f (force) option allows us to delete
   # the registry resource even though it is ONLINE.
   @cmd = ($crsctl, 'delete', 'res', $res, '-f' );
   ($status, @out) = system_cmd_capture(@cmd);
   if ($status == 0) {
      trace ("@cmd ... success");
   } else {
      trace ("@cmd ... failed");
      return FAILED;
   }
   # delete the registry type
   @cmd = ($crsctl, 'delete', 'type', 'ora.registry.acfs.type');
   ($status, @out) = system_cmd_capture(@cmd);
   if ($status == 0) {
      trace ("@cmd ... success");
   } else {
      trace ("@cmd ... failed (benign)");
      # harmless enough. Let's not abort an upgrade for this.
      $status = SUCCESS;
   }
   
   return SUCCESS;
}

sub disableACFSDriver
#-------------------------------------------------------------------------------
# Function: Stop ACFS drivers
# Args    : 0
#-------------------------------------------------------------------------------
{
   my $crsctl = crs_exec_path('crsctl');
   my $res    = 'ora.drivers.acfs';

   # disable acfs drivers
   my @cmd    = ($crsctl, 'modify', 'resource', $res,
                 '-attr', "\"ENABLED=0\"", '-init');
   my $status = system_cmd(@cmd);

   if ($status == 0) {
      trace ("@cmd ... success");
   } else {
      trace ("@cmd ... failed");
   }
}

sub deleteACFSDriver
#-------------------------------------------------------------------------------
# Function: Delete ACFS resource and uninstall acfsroot
# Args    : 0
#-------------------------------------------------------------------------------
{
   my $crsctl = crs_exec_path('crsctl');
   my $res = 'ora.drivers.acfs';

   # delete acfs drivers
   my @cmd = ($crsctl, 'delete', 'res', $res, '-init', '-f');

   trace ("@cmd");
   my $status = system_cmd(@cmd);
   if ($status == 0) {
      trace ("@cmd ... success");
   } else {
      trace ("@cmd ... failed");
   }
}

sub removeACFSRoot
{
   my $chkpoints =  $_;
   my $acfsroot;
   my $acfsutil;
   my $has       =  "has";
   my $crsctl    =  catfile ($CFG->ORA_CRS_HOME, "bin", "crsctl");

   if ($CFG->platform_family eq 'windows') {
      $acfsroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'acfsroot.bat');
   }
   else {
      $acfsroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'acfsroot');
   }
   
   if ($CFG->platform_family eq 'windows') {
      $acfsutil = catfile ($CFG->ORA_CRS_HOME, 'bin', 'acfsutil');
   }
   else {
      $acfsutil = "/sbin/acfsutil";
   }

   if (! (-e $acfsroot)) {
      trace ("ADVM/ACFS is not configured");
      return;
   }

   if ( !$CFG->SIHA )
   {
      $has = "crs"; 
   }
   if ( $chkpoints == 1 )
   {  
      if (!isCkptexist("ROOTCRS_ACFSUNINST")) 
      {
         trace("Writing checkpoint for USM driver uninstall");
         writeCkpt("ROOTCRS_ACFSUNINST", CKPTSTART);
         $CFG->wipCkptName("ROOTCRS_ACFSUNINST");
      }
   }

   if (($chkpoints == 0) || (!isCkptSuccess("ROOTCRS_ACFSUNINST")) )
   {
      $CFG->wipCkptName("ROOTCRS_ACFSINST");

      my @cmd    = ($acfsroot, 'uninstall');
      trace ("Executing @cmd");
      my @output = system_cmd_capture(@cmd);
      my $rc = shift @output;

      if ( $rc == 0 ) {
         trace ("@cmd ... success");
         trace("ACFS drivers uninstall completed");
                 
         # Only try a re-enable if we had previously failed.
         # OR if we have no chkpoints. AND we are in the DOWNGRADE process
         # This is because we only require a reboot in the downgrade process. 
         # See below. So we disable crs before reboot and enable it here.

         if ( (($chkpoints == 0)                                        || 
               ((getCkptStatus("ROOTCRS_ACFSUNINST") == CKPTFAIL)))    &&
              ($CFG->DOWNGRADE)                                          )
         { 
            my $enable = "$crsctl enable $has";
            @output = system_cmd_capture($enable);
            $rc = shift @output;
            trace ("@cmd ... success");
            
            if ($rc != 0) {
               trace("$enable .... unable to enable CRS.");
               print_error(435);
               exit 1;
            }
            else {
               trace("$enable .... success.");
            }
         }

         if ($chkpoints == 1)
         {
            writeCkpt("ROOTCRS_ACFSUNINST", CKPTSUC);
         }

      }
      elsif ( $rc != 2 )
      {
         #We can not unload and remove the drivers,
         #We'll try to turn off the ACFS persistent log
         if (-e $acfsutil) {
            my $cmd = "$acfsutil plogconfig -t";
            if ($CFG->platform_family eq 'windows') {
               $cmd =~ s/\ -/\ \//g ;
            }
            #We'll keep going, so we won't check the output
            system_cmd_capture($cmd);
         }
          
         if ((scalar(grep(/9118/, @output))) || 
             (scalar(grep(/9119/, @output))) ||
             (scalar(grep(/9348/, @output))) > 0)
         {
            #We'll keep going if we need to reboot but we are in a development
            #environment.
            if (is_dev_env())
            {
               trace ("Skipping reboot in development env ... success");
               trace("ACFS drivers uninstall completed");
               writeCkpt("ROOTCRS_ACFSUNINST", CKPTSUC);
               return SUCCESS;
            }
            
            #
            # We couldn't unload the old drivers or load the new ones.
            trace("ACFS drivers unable to be uninstalled.");
            if ($CFG->DOWNGRADE)
            {    
               print color 'bold';
               print_error(434);
               print color 'reset';
               
               my $disable = "$crsctl disable $has";
               trace("$disable ... disabling CRS in preparation for reboot.");
               @output = system_cmd_capture($disable);
               $rc = shift @output;

               if ($rc != 0)
               {
                   trace("$disable ... unable to disable CRS for reboot.");
                   # Don't fail here - the user should correct
                   # the disable, by running it by hand, and reboot.
                   print_error(436);
               }
               else
               {
                   trace("$disable ... CRS disabled, ready for reboot.");
                   # At this point, CRS is disabled.  The user will need
                   # to reboot, rerun the operation.  This will rely on 
                   # OPatch and OUI to print the appropriate error message.
               }
            }

            print_error(205);
            if ($chkpoints == 1)
            {
               writeCkpt("ROOTCRS_ACFSUNINST", CKPTFAIL);
            }
            exit 1;
         }   
      }
      else 
      {
         trace ("@cmd ... failed");
         print_error(205);
         if ($chkpoints == 1)
         {
            writeCkpt("ROOTCRS_ACFSUNINST", CKPTFAIL);
         }
         exit 1;
      }
   }
}

=head2 acfsrepl_updateResources

   Updates the replication resources to run at the upgraded version.
   Currently this upgrades from 11.2.0.2 on Linux. For this we do the
   following:
   - Stop all the replication resources.
   - Add the attributes PRIMARY_DEVICE and MOUNTPATH to the resources.

=head3 Parameters

   None

=head3 Returns

  None 

=head3 Notes
  
  Runs only on Linux

=cut
sub acfsrepl_updateResources
{
   my $crsctlbin = catfile ($CFG->ORA_CRS_HOME, "bin", "crsctl");
   my @repltypes = ('ora.acfsrepltransport.type', 'ora.acfsreplapply.type', 
                    'ora.acfsreplmain.type', 'ora.acfsreplinit.type');
   my @replres;
   my %repldevhash;
   my $type;
   my $cmd;
   my $status;

   if ($^O !~ /linux/i) 
   {
      # Replication is only supported starting with Linux 11.2.0.2
      return;
   }

   my @oldcrs_ver = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   if (!($oldcrs_ver[0] eq '11' && $oldcrs_ver[1] eq '2' && 
       $oldcrs_ver[2] eq '0'  && $oldcrs_ver[3] eq '2') ) 
   {
     # We aren't running on 11.2.0.2
     return;
   }

   # Find all of the replication resources on this node and put them in an array.
   foreach $type (@repltypes)
   {
      # Add the resource to the list
      acfsrepl_findRes($type, \@replres, \%repldevhash);
   }

   if(@replres == 0)
   {
     # Don't do anything if we don't have any resources
     return;
   }

   foreach $type (@repltypes)
   {
      # Add the PRIMARY_DEVICE attribute to the resource type
      $cmd = "$crsctlbin modify type $type -attr \"ATTRIBUTE=PRIMARY_DEVICE\",\"TYPE=string\",\"DEFAULT_VALUE=\",\"FLAGS=CONFIG\"";
      trace("Invoking \"$cmd\"");
      $status = system_cmd ("$cmd");
   }

   # Stop the resources
   trace("Stopping replication resources");
   foreach(0..$#replres)
   {
      trace("Stopping $replres[$_]");
      $cmd = "$crsctlbin stop resource $replres[$_]";
      trace("Invoking \"$cmd\"");
      $status = system_cmd ("$cmd");
   }

   # Update the resources
   foreach(0..$#replres)
   {
      my $res = $replres[$_];
      my $dev = $repldevhash{$res};
      
      trace("Updating resource $res");
      if($dev ne "")
      {
         # Add the PRIMARY_DEVICE attribute to the resource
         $cmd = "$crsctlbin modify resource $res -attr \"PRIMARY_DEVICE=$dev\"";
         trace ("Invoking \"$cmd\"");
         $status = system_cmd ("$cmd");
      }
   }
}

=head2 acfsrepl_findRes

   Finds all the replication resources of a given type.
   Adds the resources to an array
   Adds the resources primary device to a hash list.

=head3 Parameters

   type     - Resource type to search for.
   arrayref - Reference to array that holds the resource names found.
   hashref  - Reference to hash table that hold the primary devices found.

=head3 Returns

  None 

=head3 Notes

 - Only runs on Linux.
 
=cut
sub acfsrepl_findRes
{
  my $type     = shift;
  my $arrayref = shift;
  my $hashref  = shift;

  my $crsctlbin = catfile ($CFG->ORA_CRS_HOME, "bin", "crsctl");
  my $cmd = "$crsctlbin stat res -f -w \'TYPE = $type\'";
  my $devcmd = "/sbin/acfsutil info fs -o primaryvolume"; 

  my $line;
  my $res;

  open (OUT, "$cmd | ") or trace("Failed:$cmd($!)");
  while (<OUT>)
  {
    $line = $_;
    if($line =~ /^NAME=/)
    {
      $res = $';
      chomp($res);
      push(@$arrayref, $res);
    }
    if($line =~ /^MOUNTPOINT=/)
    {
      my $dev = `$devcmd $' 2>&1`;
      chomp($dev);
      ${$hashref}{ $res } = $dev;
    }
  }

}

=head2 acfsrepl_lastNode

   Does the replication upgrade processing on the last node.
   This is run AFTER the active version has been switched.
   Only runs on Linux upgrading from 11.2.0.2

=head3 Parameters

  Does the following:
  - Deletes the 'main' resource
  - Adds the new preapply and monitor types.
  - Gets all the replication resources and starts them

=head3 Returns

  None 

=head3 Notes

 - Only runs on Linux.
 
=cut

sub acfsrepl_lastNode
{
   my $crsctlbin        = catfile ($CFG->ORA_CRS_HOME, "bin", "crsctl");

   my $replpreapplytype = "ora.acfsreplpreapply.type";
   my $replmonitortype  = "ora.acfsreplmonitor.type";

   my @replres;
   my @repltypes = ('ora.acfsrepltransport.type', 'ora.acfsreplapply.type', 
                    'ora.acfsreplmain.type', 'ora.acfsreplinit.type');

   my %repldevhash;
   my $res;
   my $cmd;
   my $status;
   my $type;

   my $rc;
   my @lsnodes;
   my $node;
   
   if ($^O !~ /linux/i) 
   {
      # Replication is only support on Linux 11.2.0.2
      return;
   }

   my @oldcrs_ver = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   if (!($oldcrs_ver[0] eq '11' && $oldcrs_ver[1] eq '2' && 
       $oldcrs_ver[2] eq '0'  && $oldcrs_ver[3] eq '2') ) 
   {
     # We aren't running on 11.2.0.2
     return;
   }

   # Get the node list in the cluster
   ($rc, @lsnodes) = get_olsnodes_info($CFG->OLD_CRS_HOME); 

   # Find all of the replication resources on this node and put them in an array.
   foreach $type (@repltypes)
   {
      # Add the resource to the list
      acfsrepl_findRes($type, \@replres, \%repldevhash);
   }

   if(@replres == 0)
   {
     # Don't do anything if we don't have any resources
     return;
   }

   # Add the new resource types. monitor and preapply.
   my $addmoncmd = "$crsctlbin add type $replmonitortype -basetype cluster_resource -attr \"ATTRIBUTE=MOUNTPOINT\",\"TYPE=string\",\"DEFAULT_VALUE=\",\"ATTRIBUTE=PRIMARY_DEVICE\",\"TYPE=string\", \"DEFAULT_VALUE=\",\"ATTRIBUTE=AGENT_FILENAME\",\"TYPE=string\",\"DEFAULT_VALUE=%CRS_HOME%/bin/oraagent%CRS_EXE_SUFFIX%\",\"ATTRIBUTE=AUTO_START\",\"TYPE=string\",\"DEFAULT_VALUE=never\",\"ATTRIBUTE=PLACEMENT\",\"TYPE=string\",\"DEFAULT_VALUE=restricted\",\"ATTRIBUTE=SERVER_POOLS\",\"TYPE=string\", \"DEFAULT_VALUE=*\",\"ATTRIBUTE=SITE\",\"TYPE=string\", \"DEFAULT_VALUE=\",\"ATTRIBUTE=START_TIMEOUT\",\"TYPE=int\", \"DEFAULT_VALUE=120\"";
   trace ("Invoking \"$addmoncmd\"");
   $status = system_cmd ("$addmoncmd");


   my $addpreappcmd = "$crsctlbin add type $replpreapplytype -basetype cluster_resource -attr \"ATTRIBUTE=MOUNTPOINT\",\"TYPE=string\",\"DEFAULT_VALUE=\",\"ATTRIBUTE=PRIMARY_DEVICE\",\"TYPE=string\", \"DEFAULT_VALUE=\",\"ATTRIBUTE=AGENT_FILENAME\",\"TYPE=string\",\"DEFAULT_VALUE=%CRS_HOME%/bin/oraagent%CRS_EXE_SUFFIX%\",\"ATTRIBUTE=AUTO_START\",\"TYPE=string\",\"DEFAULT_VALUE=never\",\"ATTRIBUTE=PLACEMENT\",\"TYPE=string\",\"DEFAULT_VALUE=restricted\",\"ATTRIBUTE=SERVER_POOLS\",\"TYPE=string\", \"DEFAULT_VALUE=*\",\"ATTRIBUTE=SITE\",\"TYPE=string\", \"DEFAULT_VALUE=\",\"ATTRIBUTE=START_TIMEOUT\",\"TYPE=int\", \"DEFAULT_VALUE=120\"";
   trace ("Invoking \"$addpreappcmd\"");
   $status = system_cmd ("$addpreappcmd");

   for(0..$#replres)
   {
      my $index = $_;
      $res = $replres[$index];

      if($res =~ /ora.repl.main./)
      {
         my $ressuffix = $';
         my $isStandby = 0;
         my $mntpt;
         my $pridev;
         $cmd = "$crsctlbin stat res $res -f";

         trace("Invoking \"$cmd\"");
         open (OUT, "$cmd | ") or die "Cannot run command:$cmd($!)\n";
         while (<OUT>)
         {
           if($_ =~ /^SITE=standby/)
           {
             $isStandby = 1;
           }
           if($_ =~ /^MOUNTPOINT=/)
           {
             $mntpt = $';
             chomp($mntpt);
           }
           if($_ =~ /PRIMARY_DEVICE=/)
           {
             $pridev = $';
             chomp($pridev);
           }
         }

         # Delete the main resource from the array.
         delete $replres[$index];

         # Add the new resources
         if($isStandby)
         {
           acfsrepl_addResource("ora.repl.preapply." . $ressuffix, $replpreapplytype, $mntpt, $pridev, $isStandby);
           push(@replres, "ora.repl.preapply." . $ressuffix);
         }
         acfsrepl_addResource("ora.repl.monitor." . $ressuffix,  $replmonitortype, $mntpt, $pridev, $isStandby);
         push(@replres, "ora.repl.monitor." . $ressuffix);

         # Delete the main resource
         $cmd = "$crsctlbin delete resource $res";
         trace("Invoking \"$cmd\"");
         $status = system_cmd ("$cmd");
    }

  }

  # Start all of the resources in the list
  foreach(0..$#replres)
  {
    my $res = $replres[$_];
    if(defined($res))
    {
      # Start the transport daemon on each node.
      if($res =~ /ora.repl.transport./)
      {
         foreach $node (@lsnodes)
         {
           $cmd = "$crsctlbin start resource $res -n $node";
           trace("Invoking \"$cmd\"");
           $status = system_cmd ("$cmd");
         }
      }
      else
      {
        $cmd = "$crsctlbin start resource $res";
        trace("Invoking \"$cmd\"");
        $status = system_cmd ("$cmd");
      }
    }
  }
 
}

sub acfsrepl_addResource
{
  my $resname   = shift;
  my $type      = shift;
  my $mntpt     = shift;
  my $pridev    = shift;
  my $isStandby = shift;
  my $site;
  my $cmd;
  my $status;
  my $crsctlbin = catfile ($CFG->ORA_CRS_HOME, "bin", "crsctl");

  if($isStandby == 1)
  {
    $site = "standby";
  }
  else
  {
    $site = "primary";
  }

  # Add the resource
  $cmd = "$crsctlbin add resource $resname -type $type -attr \"MOUNTPOINT=$mntpt\",\"PRIMARY_DEVICE=$pridev\",\"SITE=$site\"";
  trace("Invoking \"$cmd\"");
  $status = system_cmd ("$cmd");
}

=head2 actionACFSDriversResource

   Create/Start/Delete the ACFS Drivers Resource. 

=head3 Parameters

   $action - "add"/"start"/"delete" actions for the drivers resource

=head3 Returns

  TRUE  - Creation/Start/Delete successful
  FALSE - Creation/Start/Delete failed

=head3 Notes


=cut
sub actionACFSDriversResource
{
  my $action  = shift;
  my $crsctl  = catfile ($CFG->ORA_CRS_HOME, "bin", "crsctl");
  my $ret     = FAILED;

  if (isACFSSupported()) 
  { 
    my @out  = system_cmd_capture($crsctl, "stat", "res", "ora.drivers.acfs",
                                  "-init");
    my $notExists = grep (/CRS-2613/i, @out);
    my $status    = grep (/TARGET/i, @out);

    if (scalar($notExists) > 0 && 
        ($action eq "add" || $action eq "start"))
    {
      my $user          = $CFG->params('ORACLE_OWNER');;
      my $asmgrp        = $CFG->params('ORA_ASM_GROUP');
      my $asmowner      = $user;
      
      if ($CFG->platform_family ne "windows")
      {
        $asmowner = $CFG->SUPERUSER;
      }
      
      # Handle Windows case
      if ($CFG->platform_family eq "windows") 
      {
        my $NT_AUTHORITY = '';
        if (is_dev_env()) 
        {
          $NT_AUTHORITY = $CFG->params('ORACLE_OWNER');
        }
        
        if (! $NT_AUTHORITY ) 
        {
          $asmowner =  '';  
          $user     =  '';
          $asmgrp   =  '';
        }
        else
        {
          $asmowner =  $NT_AUTHORITY;  
          $user     =  $NT_AUTHORITY;
        }
      }

      my @acfs_attr = 
          ("ACL='owner:$asmowner:rwx,pgrp:$asmgrp:r-x,other::r--," .
           "user:$user:r-x'");
      $ret = crsohasd::crsctlResource("add", 
                                      "ora.drivers.acfs", 
                                      "ora.drivers.acfs.type", \@acfs_attr);
    }
    elsif (scalar($status) > 0  && $action eq "delete")
    {
      $ret = crsohasd::crsctlResource($action, "ora.drivers.acfs");
    }
    else
    {
      # Make sure 'crsctl stat res' command ran properly
      if (scalar($status) > 0 || scalar($notExists) > 0) 
      {
        # The resource already exists or was already deleted  
        $ret = SUCCESS; 
      }
      else
      { 
        # There was an error running the command
        $ret = FAILED; 
      }
    }

    if ($action eq "start")
    {
      # Get the status again just to be sure  
      @out  = system_cmd_capture($crsctl, "stat", "res", "ora.drivers.acfs",
                                 "-init");
      my $statusOnline  = grep (/TARGET=ONLINE/i,  @out);
      my $statusOffline = grep (/TARGET=OFFLINE/i, @out);
      
      if (scalar($statusOnline) > 0) 
      {
        # The resource has been started already  
        $ret = SUCCESS; 
      }
      elsif (scalar($statusOffline) > 0 )
      { 
        # The resource has not been started 
        $ret = crsohasd::crsctlResource($action, "ora.drivers.acfs");
      }
      else
      {
        # The command failed or resource was not created
        $ret = FAILED;
      }
    }
  }
  else {
     # if ACFS Drivers are not supported, nothing to do
     # We return SUCCESS to continue.
     $ret = SUCCESS;
  }
  return $ret;
}
